// make sure glfw does not include OpenGL header from development environment
// load glad first
#include <glad/glad.h>
#include <GLFW/glfw3.h>

#include "dpLib.h"
#include "dpUtils.h"

#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>

#include <iostream>
#include <cstdlib>
#include <cstdio>
#include <algorithm>
#include <string>
#include <filesystem>

#define STB_IMAGE_IMPLEMENTATION
#include "stb_image.h"

#ifdef DPLIB_WINDOWS
#define NOMINMAX // disable min/max macros
#include <Windows.h>
#endif

//#define DP_USE_GL_TEXTURE_RECTANGLE

typedef struct dpFbo
{
    int samples;
    int width;
    int height;
    int fbo, temp_fbo, current_fbo;
    int scene_color_tex, scene_depth_tex, temp_tex, rect_tex;
    dpFbo() : samples(0), width(0), height(0), fbo(0), temp_fbo(0), scene_color_tex(0), scene_depth_tex(0), temp_tex(0), current_fbo(-1) {}
} dpFbo;

static const char* vertex_shader_skybox_text =
"#version 330 core\n"
"layout(location = 0) in vec3 vertex;\n"
"out vec3 texCoord;\n"
"uniform mat4 PVM;\n"
"void main() {\n"
"    gl_Position = PVM * vec4(vertex, 1.0);\n"
"    texCoord = vec3(vertex.xy, -vertex.z);\n"
"}\n";

static const char* fragment_shader_skybox_text =
"#version 330 core\n"
"in vec3 texCoord;\n"
"out vec4 fragColor;\n"
"uniform samplerCube cubemap;\n"
"void main(void) {\n"
"    fragColor = texture(cubemap, texCoord);\n"
"}\n";

dpContext*  CTX = nullptr;
int         g_windowPosX = 0;
int         g_windowPosY = 0;
int         g_windowWidth = 1000;
int         g_windowHeight = 1000;
#ifdef WIN32
std::string g_configfile = "data/config.xml";
#else //def WIN32
std::string g_configfile = "data/config_linux.xml";
#endif //def WIN32
bool        g_bDebug = false;
int         g_channel = 0;

float       g_fWarping = 1.f;
float       g_fBlending = 1.f;
float       g_fBlackLevel = 1.f;
float       g_fSecondaryBlending = 1.f;

float       g_fPositionX = 0.f;
float       g_fPositionY = 0.f;

bool        g_bWarping = true;
bool        g_bBlending = true;
bool        g_bBlackLevel = true;
bool        g_bSecondaryBlending = true;

bool dpCreateFbo(int samples, int width, int height, dpFbo** pFbo);
void dpResizeFbo(int width, int height, dpFbo* pFbo);
void dpFreeFbo(dpFbo* pFbo);
void dpBindFbo(dpFbo* pFbo);
void dpUnbindFbo(dpFbo* pFbo);
GLuint dpGetColorTexture(dpFbo* pFbo);
GLuint dpGetDepthTexture(dpFbo* pFbo);
GLuint dpGetColorTextureRect(dpFbo* pFbo);
int dpGetWidth(dpFbo* pFbo);
int dpGetHeight(dpFbo* pFbo);
GLuint dpCreateShader(const char* vert, const char* frag);
//GLuint dpPngTextureLoad(const char* filename);
unsigned int dpLoadCubemap(std::vector<std::string> faces);

// OpenGL 4
#if 0
void APIENTRY GLMessageCallback(GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* message, const void* userParam)
{
    fprintf(stderr, "GL CALLBACK: %s type = 0x%x, severity = 0x%x, message = %s\n", (type == GL_DEBUG_TYPE_ERROR ? "** GL ERROR **" : ""), type, severity, message);
}
#endif

static void error_callback(int error, const char* description)
{
    fprintf(stderr, "Error: %s\n", description);
}

static void key_callback(GLFWwindow* window, int key, int scancode, int action, int mods)
{
    if (key == GLFW_KEY_ESCAPE && action == GLFW_PRESS)
    {
        glfwSetWindowShouldClose(window, GLFW_TRUE);
        fprintf(stdout, "closing window...\n");
    }

    if (action == GLFW_PRESS)
    {
        switch (key)
        {
        case GLFW_KEY_1:
            g_fWarping = std::min(g_fWarping + 0.1f, 1.f);
            break;
        case GLFW_KEY_2:
            g_fWarping = std::max(g_fWarping - 0.1f, 0.f);
            break;
        case GLFW_KEY_3:
            g_fBlending = std::min(g_fBlending + 0.1f, 1.f);
            break;
        case GLFW_KEY_4:
            g_fBlending = std::max(g_fBlending - 0.1f, 0.f);
            break;
        case GLFW_KEY_5:
            g_fBlackLevel = g_fBlackLevel + 0.1f;
            break;
        case GLFW_KEY_6:
            g_fBlackLevel = std::max(g_fBlackLevel - 0.1f, 0.f);
            break;
        case GLFW_KEY_7:
            g_fSecondaryBlending = std::min(g_fSecondaryBlending + 0.1f, 1.f);
            break;
        case GLFW_KEY_8:
            g_fSecondaryBlending = std::max(g_fSecondaryBlending - 0.1f, 0.f);
            break;
        case GLFW_KEY_Q:
            g_bWarping = !g_bWarping;
            break;
        case GLFW_KEY_W:
            g_bBlending = !g_bBlending;
            break;
        case GLFW_KEY_E:
            g_bBlackLevel = !g_bBlackLevel;
            break;
        case GLFW_KEY_R:
            g_bSecondaryBlending = !g_bSecondaryBlending;
            break;
        case GLFW_KEY_LEFT:
            g_fPositionX -= 10.f;
            break;
        case GLFW_KEY_RIGHT:
            g_fPositionX += 10.f;
            break;
        case GLFW_KEY_UP:
            g_fPositionY += 10.f;
            break;
        case GLFW_KEY_DOWN:
            g_fPositionY -= 10.f;
            break;
        }
    }
}

int main(int argc, char** argv)
{
    int i;
    for (i = 1; i < argc; i++)
    {
        std::string p(argv[i]);
        if (p[0] == '-')
        {
            std::string param = p.substr(1, std::string::npos);
            if (param == "x" && i < argc - 1)
            {
                g_windowPosX = atoi(argv[i + 1]);
                i++;
            }
            if (param == "y" && i < argc - 1)
            {
                g_windowPosY = atoi(argv[i + 1]);
                i++;
            }
            if (param == "w" && i < argc - 1)
            {
                g_windowWidth = atoi(argv[i + 1]);
                i++;
            }
            if (param == "h" && i < argc - 1)
            {
                g_windowHeight = atoi(argv[i + 1]);
                i++;
            }
            if (param == "c" && i < argc - 1)
            {
                g_channel = atoi(argv[i + 1]);
                i++;
            }
            if (param == "v" && i < argc - 1)
            {
                g_bDebug = (bool)atoi(argv[i + 1]);
                i++;
            }
        }
        else
        {
            g_configfile = argv[i];
        }
    }

    GLFWwindow* window;
    GLuint vertex_buffer_triangle, vertex_shader_triangle, fragment_shader_triangle, program_triangle;
    GLint mvp_location_triangle, vpos_location_triangle, vcol_location_triangle;
    dpFbo* framebuffer = nullptr;

    glfwSetErrorCallback(error_callback);

    if (!glfwInit())
        exit(EXIT_FAILURE);

    glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
    glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
    glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
    glfwWindowHint(GLFW_DECORATED, GLFW_FALSE);
    if (g_bDebug)
    {
        glfwWindowHint(GLFW_OPENGL_DEBUG_CONTEXT, GL_TRUE);
    }

    window = glfwCreateWindow(g_windowWidth, g_windowHeight, "Simple example", nullptr, nullptr);
    if (!window)
    {
        glfwTerminate();
        exit(EXIT_FAILURE);
    }

    if (g_bDebug)
    {
        fprintf(stdout, "glwf opengl version=%d.%d\n", glfwGetWindowAttrib(window, GLFW_CONTEXT_VERSION_MAJOR), glfwGetWindowAttrib(window, GLFW_CONTEXT_VERSION_MINOR));
    }

    glfwSetWindowPos(window, g_windowPosX, g_windowPosY);

    glfwSetKeyCallback(window, key_callback);

    glfwMakeContextCurrent(window);
    gladLoadGLLoader((GLADloadproc)glfwGetProcAddress);
    glfwSwapInterval(1);

    // OpenGL 4
#if 0
#ifdef _DEBUG
    glEnable(GL_DEBUG_OUTPUT);
    glDebugMessageCallback(GLMessageCallback, nullptr);
#endif
#endif

    int numSamples;
#ifdef DP_USE_GL_TEXTURE_RECTANGLE
    numSamples = 1;
#else
    numSamples = 1;
#endif
    if (!dpCreateFbo(numSamples, g_windowWidth, g_windowHeight, &framebuffer))
    {
        glfwTerminate();
        exit(EXIT_FAILURE);
    }

    //float skyboxVertices[] = {
    //    -1.0,  1.0,  1.0,
    //    -1.0, -1.0,  1.0,
    //     1.0, -1.0,  1.0,
    //     1.0,  1.0,  1.0,
    //    -1.0,  1.0, -1.0,
    //    -1.0, -1.0, -1.0,
    //     1.0, -1.0, -1.0,
    //     1.0,  1.0, -1.0
    //};
    float skyboxVertices[] = {
        -10.0f,  10.0f, -10.0f,
        -10.0f, -10.0f, -10.0f,
        10.0f, -10.0f, -10.0f,
        10.0f, -10.0f, -10.0f,
        10.0f,  10.0f, -10.0f,
        -10.0f,  10.0f, -10.0f,

        -10.0f, -10.0f,  10.0f,
        -10.0f, -10.0f, -10.0f,
        -10.0f,  10.0f, -10.0f,
        -10.0f,  10.0f, -10.0f,
        -10.0f,  10.0f,  10.0f,
        -10.0f, -10.0f,  10.0f,

        10.0f, -10.0f, -10.0f,
        10.0f, -10.0f,  10.0f,
        10.0f,  10.0f,  10.0f,
        10.0f,  10.0f,  10.0f,
        10.0f,  10.0f, -10.0f,
        10.0f, -10.0f, -10.0f,

        -10.0f, -10.0f,  10.0f,
        -10.0f,  10.0f,  10.0f,
        10.0f,  10.0f,  10.0f,
        10.0f,  10.0f,  10.0f,
        10.0f, -10.0f,  10.0f,
        -10.0f, -10.0f,  10.0f,

        -10.0f,  10.0f, -10.0f,
        10.0f,  10.0f, -10.0f,
        10.0f,  10.0f,  10.0f,
        10.0f,  10.0f,  10.0f,
        -10.0f,  10.0f,  10.0f,
        -10.0f,  10.0f, -10.0f,

        -10.0f, -10.0f, -10.0f,
        -10.0f, -10.0f,  10.0f,
        10.0f, -10.0f, -10.0f,
        10.0f, -10.0f, -10.0f,
        -10.0f, -10.0f,  10.0f,
        10.0f, -10.0f,  10.0f
    };

    unsigned short skyboxIndices[] = {
        0, 1, 2, 3,
        3, 2, 6, 7,
        7, 6, 5, 4,
        4, 5, 1, 0,
        0, 3, 7, 4,
        1, 2, 6, 5
    };

    std::vector<std::string> faces
    {
        "data/cubemap/right.png",
        "data/cubemap/left.png",
        "data/cubemap/top.png",
        "data/cubemap/bottom.png",
        "data/cubemap/front.png",
        "data/cubemap/back.png"
    };

    unsigned int skyboxVAO, skyboxVBO, skyboxIBO;

    glGenBuffers(1, &skyboxVBO);
    glBindBuffer(GL_ARRAY_BUFFER, skyboxVBO);
    glBufferData(GL_ARRAY_BUFFER, 3 * 36 * sizeof(float), &skyboxVertices, GL_STATIC_DRAW);

    glGenVertexArrays(1, &skyboxVAO);
    glBindVertexArray(skyboxVAO);
    glEnableVertexAttribArray(0);
    glBindBuffer(GL_ARRAY_BUFFER, skyboxVBO);
    glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)0);

    glGenBuffers(1, &skyboxIBO);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, skyboxIBO);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(skyboxIndices), skyboxIndices, GL_STATIC_DRAW);


    unsigned int skyboxTexture = dpLoadCubemap(faces);

    GLuint program_skybox;

    program_skybox = dpCreateShader(vertex_shader_skybox_text, fragment_shader_skybox_text);

    GLuint location_skybox_cubemap, location_skybox_pvm;

    glUseProgram(program_skybox);
    location_skybox_cubemap = glGetUniformLocation(program_skybox, "cubemap");
    location_skybox_pvm = glGetUniformLocation(program_skybox, "PVM");
    glUseProgram(0);

    //

    dpResult r = dpNoError;

#ifdef DP_USE_GL_TEXTURE_RECTANGLE
    r = dpCreateContextOpenGL(&CTX, 7, dpTextureTargetRectangle);
#else
    r = dpCreateContextOpenGL(&CTX, 6, dpTextureTarget2D);
#endif
    if (r != dpNoError)
    {
        if (g_bDebug)
        {
            // this will never work. If create Context fails we have no context to get an error string from.
            //char buffer[MAX_PATH];
            //memset(&buffer[0], 0, MAX_PATH * sizeof(char));
            //dpGetErrorString(CTX, buffer, MAX_PATH);
            //fprintf(stderr, "dpCreateContextOpenGL failed with %d\n%s\n", r, buffer);
            std::cout << "dpCreateContextOpenGL failed with " << r << std::endl;
        }
        glfwTerminate();
        exit(EXIT_FAILURE);
    }

    r = dpLoadConfigurationFromFileOpenGL(CTX, g_configfile.c_str());
    if (r != dpNoError)
    {
        if (g_bDebug)
        {
            char buffer[MAX_PATH] = { 0 };
            dpGetErrorString(CTX, buffer, MAX_PATH);
            fprintf(stderr, "dpLoadConfigurationFromFileOpenGL failed with %d\n%s\n", r, buffer);
        }
        glfwTerminate();
        exit(EXIT_FAILURE);
    }

    dpSetFlipWarpmeshVerticesYOpenGL(CTX, true);
    dpSetFlipWarpmeshTexcoordsVOpenGL(CTX, true);

    size_t numFiles = 0;
    dpGetConfigurationFileNames(CTX, 0, nullptr, &numFiles);
    char *raw = (char*)malloc(numFiles * (MAX_PATH + 1));
    char** files = (char**)malloc(sizeof(*files) * numFiles);

    // Set each pointer to the start of its corresponding section of the raw buffer.
    for (i = 0; i < numFiles; i++)
    {
        files[i] = &raw[i * (MAX_PATH + 1)];
    }
    dpGetConfigurationFileNames(CTX, 0, files, &numFiles);
    if (g_bDebug)
    {
        fprintf(stdout, "configuration for channel 0\n");
        for (i = 0; i < numFiles; i++)
        {
            fprintf(stdout, "%s\n", files[i]);
        }
    }
    free(files);
    free(raw);

    dpSetClippingPlanesOpenGL(CTX, 0.1, 1000.0);
    fprintf(stdout, "dpLib::dpSetActiveChannelOpenGL()\n");
    r = dpSetActiveChannelOpenGL(CTX, g_channel, g_windowWidth, g_windowHeight);
    if (g_bDebug)
    {
        char buffer[MAX_PATH] = { 0 };
        dpGetErrorString(CTX, buffer, MAX_PATH);
        fprintf(stderr, "dpSetActiveChannelOpenGL failed with %d\n%s\n", r, buffer);
    }
    if (r != dpNoError)
    {
        if (g_bDebug)
        {
            char buffer[MAX_PATH];
            dpGetErrorString(CTX, buffer, MAX_PATH);
            fprintf(stderr, "dpSetActiveChannelOpenGL failed with %d\n%s\n", r, buffer);
        }
    }

    //r = dpUnknownError;

    fprintf(stdout, "start render loop\n");
    while (!glfwWindowShouldClose(window))
    {
        dpVec3f eyepoint(g_fPositionX, g_fPositionY, 0);
        dpVec3f orientation;
        dpMatrix4x4 projection;

        int width, height;
        glfwGetFramebufferSize(window, &width, &height);

        static bool firstframe = true;
        if (firstframe)
        {
            fprintf(stdout, "window w=%d h=%d\n", width, height);
            fprintf(stdout, "dpError=%d\n", r);
            firstframe = false;
        }

        if (r == dpNoError)
        {
            dpSetCorrectionPassOpenGL_1(CTX, dpWarpingPass, g_fWarping);
            dpSetCorrectionPassOpenGL_1(CTX, dpBlendingPass, g_fBlending);
            dpSetCorrectionPassOpenGL_1(CTX, dpBlackLevelPass, g_fBlackLevel);
            dpSetCorrectionPassOpenGL_1(CTX, dpSecondaryBlendingPass, g_fSecondaryBlending);

            dpSetCorrectionPassOpenGL(CTX, dpWarpingPass, g_bWarping);
            dpSetCorrectionPassOpenGL(CTX, dpBlendingPass, g_bBlending);
            dpSetCorrectionPassOpenGL(CTX, dpBlackLevelPass, false/*g_bBlackLevel*/);
            dpSetCorrectionPassOpenGL(CTX, dpSecondaryBlendingPass, false/*g_bSecondaryBlending*/);

            dpPreDrawOpenGL(CTX, eyepoint, &orientation, &projection);

            dpBindFbo(framebuffer);
        }
        else
        {
            glm::mat4 p = glm::perspective(glm::radians(90.f), width / (float)height, 0.1f, 10.f);
            memcpy(&projection.matrix, glm::value_ptr(p), 16 * sizeof(float));

            orientation = dpVec3f(0.f);
        }


        glViewport(0, 0, width, height);
        glClearColor(0.3f, 0.3f, 0.3f, 1.0f);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        //glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        //glDisable(GL_CULL_FACE);

        glm::mat4 mProjection;
        glm::mat4 mView = glm::mat4(1.f);
        glm::mat4 mModel = glm::mat4(1.f);

        memcpy(glm::value_ptr(mProjection), &projection.matrix, 16 * sizeof(float));

        // draw skybox as last
        glDepthMask(GL_FALSE);
        glDepthFunc(GL_LEQUAL);  // change depth function so depth test passes when values are equal to depth buffer's content
        glUseProgram(program_skybox);
        glm::mat4 mPVM, mRotX, mRotY, mRotZ;
        mRotX = glm::rotate(glm::mat4(1.f), glm::radians(-orientation.y), glm::vec3(1.f, 0.f, 0.f)); // pitch
        mRotY = glm::rotate(glm::mat4(1.f), glm::radians(orientation.x), glm::vec3(0.f, 1.f, 0.f)); // heading
        mRotZ = glm::rotate(glm::mat4(1.f), glm::radians(-orientation.z), glm::vec3(0.f, 0.f, 1.f)); // roll
        mView = mRotZ * mRotX * mRotY;

        mPVM = mProjection * mView * mModel;

        //mat4x4_rotate(view,// remove translation from the view matrix
        glUniform1i(location_skybox_cubemap, 0);
        glUniformMatrix4fv(location_skybox_pvm, 1, GL_FALSE, (const GLfloat*)glm::value_ptr(mPVM));

        // skybox cube
        glBindVertexArray(skyboxVAO);
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_CUBE_MAP, skyboxTexture);
        glDrawArrays(GL_TRIANGLES, 0, 36);
        glBindVertexArray(0);
        glUseProgram(0);
        glDepthFunc(GL_LESS); // set depth function back to default
        glDepthMask(GL_TRUE);

        if (r == dpNoError)
        {
            dpUnbindFbo(framebuffer);
            dpPostDrawOpenGL(CTX, dpGetColorTexture(framebuffer));

            GLenum error = glGetError();
            if (error != GL_NO_ERROR)
            {

            }
        }

        //glFlush();

        glfwSwapBuffers(window);
        glfwPollEvents();
    }

    glfwDestroyWindow(window);

    glfwTerminate();
    exit(EXIT_SUCCESS);
}

bool dpCreateFbo(int samples, int width, int height, dpFbo** pFbo)
{
    if (!pFbo)
    {
        return false;
    }

    dpFbo* fbo = new dpFbo;
    fbo->width = width;
    fbo->height = height;
    fbo->samples = samples;

    int target;
    GLuint boundTexture = 0;
#ifdef DP_USE_GL_TEXTURE_RECTANGLE
    target = GL_TEXTURE_RECTANGLE;
    fbo->samples = 1;
    glGetIntegerv(GL_TEXTURE_BINDING_RECTANGLE, (GLint*)&boundTexture);
#else
    target = GL_TEXTURE_2D;
    glGetIntegerv(GL_TEXTURE_BINDING_2D, (GLint*)&boundTexture);
#endif

    if (fbo->samples > 1)
    {
        glActiveTexture(GL_TEXTURE0);

        glGenTextures(1, (GLuint*)&fbo->scene_color_tex);
        glBindTexture(GL_TEXTURE_2D_MULTISAMPLE, fbo->scene_color_tex);
        glTexImage2DMultisample(GL_TEXTURE_2D_MULTISAMPLE, fbo->samples, GL_RGB, fbo->width, fbo->height, GL_TRUE);

        glGenTextures(1, (GLuint*)&fbo->scene_depth_tex);
        glBindTexture(GL_TEXTURE_2D_MULTISAMPLE, fbo->scene_depth_tex);
        glTexImage2DMultisample(GL_TEXTURE_2D_MULTISAMPLE, 4, GL_DEPTH_COMPONENT32F, fbo->width, fbo->height, GL_TRUE);

        glBindTexture(GL_TEXTURE_2D_MULTISAMPLE, 0);

        // create 2d target texture for resolving the multisampling texture
        glGenTextures(1, (GLuint*)&fbo->temp_tex);
        glBindTexture(GL_TEXTURE_2D, fbo->temp_tex);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR); // GL_NEAREST
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, width, height, 0, GL_RGB, GL_UNSIGNED_BYTE, 0);
    }
    else
    {
        glGenTextures(1, (GLuint*)&fbo->scene_color_tex);
        glBindTexture(target, fbo->scene_color_tex);
        glTexParameteri(target, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameteri(target, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_LINEAR); // GL_NEAREST
        glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexImage2D(target, 0, GL_RGB, width, height, 0, GL_RGB, GL_UNSIGNED_BYTE, 0);

        glGenTextures(1, (GLuint*)&fbo->scene_depth_tex);
        glBindTexture(target, fbo->scene_depth_tex);
        glTexParameteri(target, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameteri(target, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_LINEAR); // GL_NEAREST
        glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexImage2D( target, 0, GL_DEPTH_COMPONENT32F, width, height, 0, GL_DEPTH_COMPONENT, GL_FLOAT, 0);
    }

    glBindTexture(target, boundTexture);

    glGenFramebuffers(1, (GLuint*)&fbo->fbo);
    glBindFramebuffer(GL_FRAMEBUFFER, fbo->fbo);
    glFramebufferTexture(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, fbo->scene_color_tex, 0);
    glFramebufferTexture(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, fbo->scene_depth_tex, 0);

    const GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        printf("external rendering will fail due to FBO error: %d\n", status);
    }

    // target fbo for blit
    if (fbo->samples > 1)
    {
        glGenFramebuffers(1, (GLuint*)&fbo->temp_fbo);
        glBindFramebuffer(GL_FRAMEBUFFER, fbo->temp_fbo);
        glFramebufferTexture(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, fbo->temp_tex, 0);

        const GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
        if (GL_FRAMEBUFFER_COMPLETE != status)
        {
            printf("external rendering will fail due to FBO error: %d\n", status);
        }
    }

    glBindFramebuffer(GL_FRAMEBUFFER, 0);

    // create GL_TEXTURE_RECTANGLE target texture
    glGenTextures(1, (GLuint*)&fbo->rect_tex);

    glGetIntegerv(GL_TEXTURE_BINDING_RECTANGLE, (GLint*)&boundTexture);
    glBindTexture(GL_TEXTURE_RECTANGLE, fbo->rect_tex);

    glTexParameteri(GL_TEXTURE_RECTANGLE, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_RECTANGLE, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_RECTANGLE, GL_TEXTURE_MIN_FILTER, GL_LINEAR); // GL_NEAREST
    glTexParameteri(GL_TEXTURE_RECTANGLE, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexImage2D(GL_TEXTURE_RECTANGLE, 0, GL_RGB, width, height, 0, GL_RGB, GL_UNSIGNED_BYTE, 0);

    glBindTexture(GL_TEXTURE_RECTANGLE, boundTexture);


    *pFbo = fbo;

    return true;
}

void dpResizeFbo(int width, int height, dpFbo* pFbo)
{
    if (!pFbo)
    {
        return;
    }

    pFbo->width = width;
    pFbo->height = height;

    int target;
#ifdef DP_USE_GL_TEXTURE_RECTANGLE
    target = GL_TEXTURE_RECTANGLE;
#else
    target = GL_TEXTURE_2D;
#endif

    if (pFbo->samples > 1)
    {
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D_MULTISAMPLE, pFbo->scene_color_tex);
        glTexImage2DMultisample(GL_TEXTURE_2D_MULTISAMPLE, pFbo->samples, GL_RGB, pFbo->width, pFbo->height, GL_TRUE);
        glBindTexture(GL_TEXTURE_2D_MULTISAMPLE, pFbo->scene_depth_tex);
        glTexImage2DMultisample(GL_TEXTURE_2D_MULTISAMPLE, 4, GL_DEPTH_COMPONENT32F, pFbo->width, pFbo->height, GL_TRUE);
        glBindTexture(GL_TEXTURE_2D_MULTISAMPLE, 0);

        glBindTexture(GL_TEXTURE_2D, pFbo->temp_tex);
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, pFbo->width, pFbo->height, 0, GL_RGB, GL_UNSIGNED_BYTE, 0);
    }
    else
    {
        glBindTexture(target, pFbo->scene_color_tex);
        glTexImage2D(target, 0, GL_RGB, pFbo->width, pFbo->height, 0, GL_RGB, GL_UNSIGNED_BYTE, 0);
        glBindTexture(target, pFbo->scene_depth_tex);
        glTexImage2D(target, 0, GL_DEPTH_COMPONENT32F, pFbo->width, pFbo->height, 0, GL_DEPTH_COMPONENT, GL_FLOAT, 0);
    }

    glBindTexture(GL_TEXTURE_RECTANGLE, pFbo->rect_tex);
    glTexImage2D(GL_TEXTURE_RECTANGLE, 0, GL_RGB, pFbo->width, pFbo->height, 0, GL_RGB, GL_UNSIGNED_BYTE, 0);
}

void dpFreeFbo(dpFbo* pFbo)
{
    if (!pFbo)
    {
        return;
    }
    if (pFbo->scene_color_tex)
        glDeleteTextures(1, (GLuint*)&pFbo->scene_color_tex);
    if (pFbo->scene_depth_tex)
        glDeleteTextures(1, (GLuint*)&pFbo->scene_depth_tex);
    if (pFbo->temp_tex)
        glDeleteTextures(1, (GLuint*)&pFbo->temp_tex);
    if (pFbo->fbo)
        glDeleteFramebuffers(1, (GLuint*)&pFbo->fbo);
    if (pFbo->temp_fbo)
        glDeleteFramebuffers(1, (GLuint*)&pFbo->temp_fbo);
    if (pFbo->rect_tex)
        glDeleteTextures(1, (GLuint*)&pFbo->rect_tex);

    delete pFbo;
    pFbo = nullptr;
}

void dpBindFbo(dpFbo* pFbo)
{
    if (!pFbo)
    {
        return;
    }

    glGetIntegerv(GL_FRAMEBUFFER_BINDING, &pFbo->current_fbo);
    glBindFramebuffer(GL_FRAMEBUFFER, pFbo->fbo);
}

void dpUnbindFbo(dpFbo* pFbo)
{
    if (!pFbo)
    {
        return;
    }

    if (pFbo->samples > 1)
    {
    #if 1

        glBindFramebuffer(GL_READ_FRAMEBUFFER, pFbo->fbo);
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER, pFbo->temp_fbo);
        glBlitFramebuffer(0, 0, pFbo->width, pFbo->height,
            0, 0, pFbo->width, pFbo->height,
            GL_COLOR_BUFFER_BIT, GL_NEAREST);
        glBindFramebuffer(GL_FRAMEBUFFER, pFbo->temp_fbo);
    #else
        // OpenGL 4.5 only
        if (*glBlitNamedFramebuffer != nullptr) {
            glBlitNamedFramebuffer(pFbo->fbo, pFbo->temp_fbo, 0, 0, pFbo->width, pFbo->height, 0, 0, pFbo->width, pFbo->height, GL_COLOR_BUFFER_BIT, GL_NEAREST);
            int result = glGetError();
        }
    #endif
    }

    int readbuf, framebufbin, samplebuf;
    glGetIntegerv(GL_READ_BUFFER, &readbuf);
    glGetIntegerv(GL_READ_FRAMEBUFFER_BINDING, &framebufbin);
    glGetIntegerv(GL_SAMPLE_BUFFERS, &samplebuf);
    //    glCopyTextureSubImage2D(pFbo->rect_tex, 0, 0, 0, 0, 0, pFbo->width, pFbo->height);

    int err = glGetError();

    glBindFramebuffer(GL_FRAMEBUFFER, pFbo->current_fbo);
}

GLuint dpGetColorTexture(dpFbo* pFbo)
{
    if (!pFbo)
    {
        return 0;
    }
    if (pFbo->samples > 1)
    {
        return pFbo->temp_tex;
    }
    return pFbo->scene_color_tex;
}

GLuint dpGetDepthTexture(dpFbo* pFbo)
{
    if (!pFbo)
    {
        return 0;
    }
    return pFbo->scene_depth_tex;
}

GLuint dpGetColorTextureRect(dpFbo* pFbo)
{
    if (!pFbo)
    {
        return 0;
    }
    return pFbo->rect_tex;
}

int dpGetWidth(dpFbo* pFbo)
{
    if (!pFbo)
    {
        return 0;
    }
    return pFbo->width;
}

int dpGetHeight(dpFbo* pFbo)
{
    if (!pFbo)
    {
        return 0;
    }
    return pFbo->height;
}

GLuint dpCreateShader(const char* vert, const char* frag)
{
    GLuint VS = glCreateShader(GL_VERTEX_SHADER);
    GLuint FS = glCreateShader(GL_FRAGMENT_SHADER);
    GLint result = GL_FALSE;
    int length;
    glShaderSource(VS, 1, &vert, NULL);
    glCompileShader(VS);
    glGetShaderiv(VS, GL_COMPILE_STATUS, &result);
    glGetShaderiv(VS, GL_INFO_LOG_LENGTH, &length);
    if (result == GL_FALSE)
    {
        char* buffer = new char[length + 1];
        glGetShaderInfoLog(VS, length, NULL, &buffer[0]);
        printf("VS error: %s\n", buffer);
        delete[] buffer;
        glDeleteShader(VS);
        glDeleteShader(FS);
        return 0;
    }
    glShaderSource(FS, 1, &frag, NULL);
    glCompileShader(FS);
    glGetShaderiv(FS, GL_COMPILE_STATUS, &result);
    glGetShaderiv(FS, GL_INFO_LOG_LENGTH, &length);
    if (result == GL_FALSE)
    {
        char* buffer = new char[length + 1];
        glGetShaderInfoLog(FS, length, NULL, &buffer[0]);
        printf("VS error: %s\n", buffer);
        delete[] buffer;
        glDeleteShader(VS);
        glDeleteShader(FS);
        return 0;
    }


    GLuint program = glCreateProgram();
    glAttachShader(program, VS);
    glAttachShader(program, FS);
    glLinkProgram(program);

    glGetProgramiv(program, GL_LINK_STATUS, &result);
    glGetProgramiv(program, GL_INFO_LOG_LENGTH, &length);
    if (result == GL_FALSE)
    {
        char* buffer = new char[length + 1];
        glGetProgramInfoLog(program, length, NULL, &buffer[0]);
        printf("PROGRAM error: %s\n", buffer);
        delete[] buffer;
        glDetachShader(program, VS);
        glDetachShader(program, FS);
        glDeleteShader(VS);
        glDeleteShader(FS);
        return 0;
    }

    return program;
}

// loads a cubemap texture from 6 individual texture faces
// order:
// +X (right)
// -X (left)
// +Y (top)
// -Y (bottom)
// +Z (front) 
// -Z (back)
// -------------------------------------------------------
unsigned int dpLoadCubemap(std::vector<std::string> faces)
{
    unsigned int textureID;
    glGenTextures(1, &textureID);
    glBindTexture(GL_TEXTURE_CUBE_MAP, textureID);

    int width, height, nrChannels;
    for (unsigned int i = 0; i < faces.size(); i++)
    {
        unsigned char *data = stbi_load(faces[i].c_str(), &width, &height, &nrChannels, 0);
        if (data)
        {
            glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGB, width, height, 0, GL_RGB, GL_UNSIGNED_BYTE, data);
            stbi_image_free(data);
        }
        else
        {
            printf("Cubemap texture failed to load at path:%s\n", faces[i].c_str());
            stbi_image_free(data);
        }
    }
    glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);

    return textureID;
}
